#include "libfma.h"
#include "lf_fabric.h"
#include "lf_topo_map.h"

#include "fma.h"
#include "fma_fabric.h"
#include "fma_myri.h"
#include "fma_tunnel.h"
#include "fma_map_fabric.h"

static void fma_free_nic(struct lf_nic *nicp);
static void fma_free_xbar(struct lf_xbar *xp);
static void fma_free_host(struct lf_host *hp);
static void fma_free_fabric(struct lf_fabric *fp);

struct lf_xbar *
fma_create_xbar(
  int num_ports,
  int xbar_id,
  int quadrant_disable)
{
  struct lf_xbar *xp;

  xp = lf_alloc_xbar(num_ports);
  if (xp == NULL) LF_ERROR(("Allocating xbar"));

  LF_CALLOC(xp->user.fma, struct fma_xbar, 1);
  FMA_XBAR(xp)->parent = xp;
  LF_CALLOC(FMA_XBAR(xp)->dfs_link_load, int, xp->num_ports);
  LF_CALLOC(FMA_XBAR(xp)->dfs_port_order, char, xp->num_ports);

  LF_CALLOC(FMA_XBAR(xp)->xbar_dist, signed char, A.myri->nic_ports);

  xp->xbar_id = xbar_id;
  xp->quadrant_disable = quadrant_disable;

  /* set up destructor */
  xp->xbar_user_destructor = fma_free_xbar;

  return xp;

 except:
  return NULL;
}

static void
fma_free_xbar(
  struct lf_xbar *xp)
{
  LF_FREE(FMA_XBAR(xp)->dfs_link_load);
  LF_FREE(FMA_XBAR(xp)->dfs_port_order);
  LF_FREE(FMA_XBAR(xp)->mf_info);
  LF_FREE(FMA_XBAR(xp)->xbar_dist);
  LF_FREE(xp->user.fma);
}

struct lf_nic *
fma_create_nic(
  int num_ports,
  lf_mac_addr_t mac_addr,
  enum lf_firmware_type fw_type)
{
  struct lf_nic *nicp;

  nicp = lf_alloc_nic(num_ports);
  if (nicp == NULL) LF_ERROR(("Allocating nic"));

  LF_MAC_COPY(nicp->mac_addr, mac_addr);

  LF_CALLOC(nicp->user.fma, struct fma_nic, 1);
  FMA_NIC(nicp)->parent = nicp;
  FMA_NIC(nicp)->fw_type = fw_type;

  LF_CALLOC(FMA_NIC(nicp)->dfs_route_count, unsigned char,
      LF_MAX_NIC_PORTS * A.myri->nic_ports);

  /* set up destructor */
  nicp->nic_user_destructor = fma_free_nic;

  return nicp;

 except:
  fma_perror_exit(1);
  return NULL;
}

static void
fma_free_nic(
  struct lf_nic *nicp)
{
  /* free mapper specific things */
  fma_mf_free_nic(nicp);

  /* free all the user stuff */
  LF_FREE(FMA_NIC(nicp)->route_buf);
  LF_FREE(FMA_NIC(nicp)->route_lens);
  LF_FREE(FMA_NIC(nicp)->dfs_route_count);
  LF_FREE(nicp->user.fma);
}

struct lf_host *
fma_create_host()
{
  struct lf_host *hp;

  LF_CALLOC(hp, struct lf_host, 1);
  LF_CALLOC(hp->user.fma, struct fma_host, 1);

  hp->host_user_destructor = fma_free_host;
  return hp;

 except:
  fma_perror_exit(1);
  return NULL;
}

static void
fma_free_host(
  struct lf_host *hp)
{
  /* If map send to this host in progress, cancel it */
  if (FMA_HOST(hp)->map_send_id != 0) {
    fma_tunnel_cancel_send(FMA_HOST(hp)->map_send_id);
    FMA_HOST(hp)->map_send_id = 0;
  }

  if (A.my_host == hp) {
    A.my_host = NULL;
  }

  LF_FREE(hp->user.fma);
}

struct lf_fabric *
fma_create_fabric()
{
  struct lf_fabric *fp;

  LF_CALLOC(fp, struct lf_fabric, 1);
  LF_CALLOC(fp->user.fma, struct fma_fabric, 1);
  fp->fabric_user_destructor = fma_free_fabric;
  return fp;

 except:
  fma_perror_exit(1);
  return NULL;
}

static void
fma_free_fabric(
  struct lf_fabric *fp)
{
  /* free our NIC array */
  LF_FREE(FMA_FABRIC(fp)->nics);

  /* free link_end info */
  LF_FREE(FMA_FABRIC(fp)->link_end_node);
  LF_FREE(FMA_FABRIC(fp)->link_end_port);

  LF_FREE(fp->user.fma);
}

/*
 * Convert a mac address to a hostname if possible
 */
char *
fma_mac_to_hostname(
  lf_mac_addr_t mac_addr)
{
  static lf_string_t buf[4];
  static int si;
  struct lf_nic *nicp;
  char *s;

  /* rotate return buf so that printf() may make more than one call */
  ++si;
  if (si >= 4) si=0;
  s = buf[si];

  nicp = lf_find_nic_by_mac(A.fabric, mac_addr);
  if (nicp == NULL
      || nicp->host == NULL
      || nicp->host->hostname == NULL
      || nicp->host->hostname[0] == '\0') {
    sprintf(s, LF_MAC_FORMAT, LF_MAC_ARGS(mac_addr));
  } else {
    sprintf(s, "%s (" LF_MAC_FORMAT ")", nicp->host->hostname,
	LF_MAC_ARGS(mac_addr));
  }

  return s;
}


/*
  return in 'port_order' a shuffled list of port numbers
*/

void
fma_shuffle_port_order(
  struct lf_xbar*xp,
  char*port_order)
{
  int i;
  int num_ports = xp->num_ports;
  
  /*fill port_order array with sorted port numbers*/
  for (i = 0; i < num_ports; i++)
    port_order [i] = i;
  
  /*now shuffle them*/
  for (i = 0; i < num_ports; i++) {
    int a = i + (random () >> 4) % (num_ports - i);
    int t = port_order [a];
    port_order[a] = port_order[i];
    port_order[i] = t;
  }
}
